'use client';

import React, { useState } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Zap, Fuel, Landmark, Cpu, ArrowRight, Sparkles, TrendingUp, Shield, Layers, ShieldCheck } from 'lucide-react';
import { useLanguage } from '@/contexts/language-context';

const iconMap = {
  'Energy & Solar Development': Zap,
  'Oil & Gas Infrastructure': Fuel,
  'Project Finance & Investment': Landmark,
  'Industrial Automation & Digital Oilfields': Cpu,
  'تطوير الطاقة والطاقة الشمسية': Zap,
  'البنية التحتية للنفط والغاز': Fuel,
  'تمويل المشاريع والاستثمار': Landmark,
  'الأتمتة الصناعية وحقول النفط الرقمية': Cpu,
};

const sectorColors = [
  {
    gradient: 'from-amber-500 via-yellow-500 to-gold-400',
    bgGradient: 'from-amber-50 via-yellow-50 to-gold-50',
    accent: 'text-amber-600',
    border: 'border-amber-300/50'
  },
  {
    gradient: 'from-emerald-500 via-green-500 to-lime-400',
    bgGradient: 'from-emerald-50 via-green-50 to-lime-50',
    accent: 'text-emerald-600',
    border: 'border-emerald-300/50'
  },
  {
    gradient: 'from-golden-500 via-amber-500 to-yellow-400',
    bgGradient: 'from-golden-50 via-amber-50 to-yellow-50',
    accent: 'text-golden-600',
    border: 'border-golden-300/50'
  },
  {
    gradient: 'from-teal-500 via-green-500 to-cyan-400',
    bgGradient: 'from-teal-50 via-green-50 to-cyan-50',
    accent: 'text-teal-600',
    border: 'border-teal-300/50'
  },
];

export default function CoreSectors() {
  const { content, language } = useLanguage();
  const { coreSectors } = content;
  const [hoveredIndex, setHoveredIndex] = useState(0);

  return (
    <section id="sectors" className="scroll-mt-20 relative overflow-hidden py-20 px-4 bg-gradient-to-br from-slate-50 via-white to-amber-50/20">
      <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[800px] h-[800px] bg-gradient-to-br from-amber-500/5 via-green-500/5 to-transparent rounded-full blur-3xl"></div>

      <div className="max-w-7xl mx-auto relative z-10">
        <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
            className="text-center relative mb-16"
        >
            <div className="inline-flex items-center gap-2 mb-4 px-4 py-2 rounded-full bg-gradient-to-r from-amber-500/10 via-green-500/10 to-amber-500/10 border border-amber-500/20">
                <Layers className="w-4 h-4 text-amber-600" />
                <span className="text-sm font-medium text-primary">{language === 'ar' ? 'قطاعاتنا الأساسية' : 'Our Core Sectors'}</span>
            </div>
            <h2 className="font-headline text-3xl font-bold bg-gradient-to-r from-amber-600 via-green-600 to-amber-600 bg-clip-text text-transparent sm:text-4xl">
                {coreSectors.title}
            </h2>
            <div className="mt-4 mx-auto w-24 h-1 bg-gradient-to-r from-transparent via-green-500 to-transparent rounded-full"></div>
        </motion.div>

        <div className="grid grid-cols-1 items-start gap-12 lg:grid-cols-12">
          {/* Left Side - Info Panel */}
          <div className="lg:col-span-5 lg:sticky lg:top-24">
            <AnimatePresence mode="wait">
              <motion.div
                key={hoveredIndex}
                initial={{ opacity: 0, y: 20, scale: 0.95 }}
                animate={{ opacity: 1, y: 0, scale: 1 }}
                exit={{ opacity: 0, y: -20, scale: 0.95 }}
                transition={{ duration: 0.4 }}
                className="relative group"
              >
                <motion.div
                  className={`absolute -inset-2 bg-gradient-to-r ${sectorColors[hoveredIndex % sectorColors.length].gradient} rounded-3xl blur-2xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
                  animate={{ scale: [1, 1.05, 1] }}
                  transition={{ duration: 3, repeat: Infinity }}
                />
                <div className={`relative overflow-hidden border-2 ${sectorColors[hoveredIndex % sectorColors.length].border} shadow-2xl bg-gradient-to-br from-white via-white ${sectorColors[hoveredIndex % sectorColors.length].bgGradient} rounded-3xl p-8`}>
                  <div className="relative z-10">
                    <motion.div
                      initial={{ scale: 0, rotate: -180 }}
                      animate={{ scale: 1, rotate: 0 }}
                      transition={{ duration: 0.5, type: "spring" }}
                      className="inline-flex mb-6"
                    >
                      <div className="relative">
                        <div className={`relative w-20 h-20 rounded-2xl bg-gradient-to-br ${sectorColors[hoveredIndex % sectorColors.length].gradient} flex items-center justify-center shadow-xl`}>
                          {(() => {
                            const Icon = iconMap[coreSectors.items[hoveredIndex].title] || Zap;
                            return (
                              <motion.div
                                animate={{ rotate: [0, 5, -5, 0] }}
                                transition={{ duration: 2, repeat: Infinity, repeatDelay: 1 }}
                              >
                                <Icon className="w-10 h-10 text-white" />
                              </motion.div>
                            );
                          })()}
                        </div>
                      </div>
                    </motion.div>
                    <h3 className="text-2xl font-bold mb-4 text-slate-900">
                      {coreSectors.items[hoveredIndex].title}
                    </h3>
                    <p className="text-base text-slate-600 leading-relaxed min-h-[100px]">
                      {coreSectors.items[hoveredIndex].description}
                    </p>
                    <div className="flex items-center justify-between pt-6 border-t-2 border-slate-200/50 mt-6">
                      <div className="flex items-center gap-2">
                         <ShieldCheck className={`w-5 h-5 ${sectorColors[hoveredIndex % sectorColors.length].accent}`} />
                         <span className="text-sm font-semibold text-slate-600">{language === 'ar' ? 'خبرة متخصصة' : 'Specialized Expertise'}</span>
                      </div>
                      <motion.div
                        animate={{ x: [0, 5, 0] }}
                        transition={{ duration: 1.5, repeat: Infinity }}
                        className={`w-12 h-12 rounded-xl bg-gradient-to-br ${sectorColors[hoveredIndex % sectorColors.length].gradient} flex items-center justify-center shadow-lg`}
                      >
                        <ArrowRight className="w-6 h-6 text-white" />
                      </motion.div>
                    </div>
                  </div>
                </div>
              </motion.div>
            </AnimatePresence>
          </div>

          {/* Right Side - Sectors Grid */}
          <div className="lg:col-span-7">
            <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
              {coreSectors.items.map((sector, index) => {
                const Icon = iconMap[sector.title] || Zap;
                const isHovered = hoveredIndex === index;
                const colors = sectorColors[index % sectorColors.length];

                return (
                  <motion.div
                    key={sector.title}
                    initial={{ opacity: 0, scale: 0.9 }}
                    whileInView={{ opacity: 1, scale: 1 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.4, delay: index * 0.1 }}
                    onMouseEnter={() => setHoveredIndex(index)}
                    className="relative group"
                  >
                    <motion.div
                      className={`absolute -inset-2 bg-gradient-to-r ${colors.gradient} rounded-3xl blur-xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
                      animate={isHovered ? { scale: [1, 1.05, 1] } : {}}
                      transition={{ duration: 2, repeat: Infinity }}
                    />
                    <div className={`relative cursor-pointer bg-white/80 backdrop-blur-sm p-6 rounded-3xl border-2 ${isHovered ? colors.border : 'border-slate-200/50'} shadow-xl hover:shadow-2xl transition-all duration-500 overflow-hidden group-hover:-translate-y-2`}>
                      <motion.div
                        className={`absolute inset-0 bg-gradient-to-br ${colors.bgGradient} opacity-0 group-hover:opacity-100 transition-opacity duration-700`}
                      />
                      <div className="relative z-10 flex items-start gap-4">
                        <motion.div
                          animate={{ scale: isHovered ? 1.1 : 1 }}
                          transition={{ duration: 0.5 }}
                          className="relative flex-shrink-0"
                        >
                          <div className={`relative flex h-16 w-16 items-center justify-center rounded-2xl bg-gradient-to-br ${isHovered ? colors.gradient : 'from-slate-100 to-slate-200'} shadow-lg transition-all duration-500`}>
                            <Icon className={`h-8 w-8 ${isHovered ? 'text-white' : 'text-slate-700'} transition-all duration-500`} />
                          </div>
                        </motion.div>
                        <div className="flex-1 min-w-0">
                          <h3 className={`text-lg font-bold mb-2 transition-colors duration-300 ${isHovered ? colors.accent : 'text-slate-900'}`}>
                            {sector.title}
                          </h3>
                          <motion.div
                            initial={{ opacity: 0, y: 10 }}
                            animate={{ opacity: isHovered ? 1 : 0, y: isHovered ? 0 : 10 }}
                            transition={{ duration: 0.3 }}
                            className={`inline-flex items-center gap-1.5 px-3 py-1 rounded-lg bg-gradient-to-r ${colors.gradient} text-white text-xs font-semibold shadow-md`}
                          >
                            <Sparkles className="w-3 h-3" />
                            <span>{language === 'ar' ? 'مجال رئيسي' : 'Key Area'}</span>
                          </motion.div>
                        </div>
                      </div>
                      <motion.div
                        className={`absolute bottom-0 left-0 right-0 h-1 bg-gradient-to-r ${colors.gradient}`}
                        initial={{ scaleX: 0 }}
                        animate={{ scaleX: isHovered ? 1 : 0 }}
                        transition={{ duration: 0.5 }}
                        style={{ transformOrigin: 'left' }}
                      />
                    </div>
                  </motion.div>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}
    